<?php

class MusicHandler
{

    // L'adresse IP de l'enceinte connectée
    private $ip;

    // La socket utilisée pour communiquer avec l'enceinte
    private $socket;

    // Le port de l'enceinte connectée
    private static $PORT = 5046;

    // Le port de l'enceinte connectée
    private static $RETRIES = 3;

    // Liste de commandes
    private static $CMD_VOLUME = 'V';

    private static $CMD_LOCATE = 'L';

    private static $CMD_RESET = 'R';

    private static $RESPONSE_NOK = 'N';

    public function __construct($ip)
    {
        $this->ip = $ip;
    }

    /**
     * Augmente le volume d'un incrément
     *
     * @return boolean
     */
    public function volumeUp()
    {
        return $this->sendCommand(self::$CMD_VOLUME . "+");
    }

    /**
     * Baisse le volume d'un incrément
     *
     * @return boolean
     */
    public function volumeDown()
    {
        return $this->sendCommand(self::$CMD_VOLUME . "-");
    }

    /**
     *
     * @return boolean
     */
    public function setVolume($volume)
    {
        if ($volume < 0 || $volume > 100) {
            return false;
        }
        return $this->sendCommand(self::$CMD_VOLUME . $volume);
    }

    public function getVolume(){
        return $this->sendCommand(self::$CMD_VOLUME);
    }
    
    /**
     *
     * @return boolean
     */
    public function doReset()
    {
        return $this->sendCommand(self::$CMD_RESET);
    }
    
    public function doLocate(){
        return $this->sendCommand(self::$CMD_LOCATE);
    }

    /**
     * Envoie une commande au module
     *
     * @param string $message
     */
    private function sendCommand($message)
    {
        $result = false;
        if ($this->openSocket() !== FALSE) {
            if (socket_sendto($this->socket, $message, strlen($message), 0, $this->ip, self::$PORT) !== FALSE) {
                $result = $this->readAcknoledge();
            }
        }
        $this->closeSocket();
        return $result;
    }

    /**
     * Attend la confirmation de la commande
     *
     * @return boolean
     */
    private function readAcknoledge()
    {
        $buf = "";
        $retries = 0;
        while ($retries < self::$RETRIES) {
            $byteReceived = socket_recvfrom($this->socket, $buf, 2048, 0, $this->ip, self::$PORT);
            if ($byteReceived > 0) {
                return $buf != self::$RESPONSE_NOK ? $buf : FALSE;
            }
            usleep(500000);
            $retries ++;
        }
        return false;
    }

    /**
     * Ouverture de la socket
     */
    private function openSocket()
    {
        if (($this->socket = socket_create(AF_INET, SOCK_DGRAM, SOL_UDP)) === FALSE) {
            $errorcode = socket_last_error();
            $errormsg = socket_strerror($errorcode);
            echo ("Couldn't create socket: [$errorcode] $errormsg \n");
            return false;
        }
        return true;
    }

    /**
     * Fermeture de la socket
     */
    private function closeSocket()
    {
        socket_close($this->socket);
    }
}