<?php
abstract class Daemon {
	static public $PROCESS_NAME;
	protected $name;
	private $isRunning = true;
	private $signals = array (
			SIGTERM,
			SIGINT,
			SIGCHLD,
			SIGHUP 
	);
	/**
	 * Return an array with all PIDs or false
	 *
	 * @param string $process_name
	 *        	<br> the process name
	 * @return array|boolean
	 */
	public static function getPIDs() {
		$cmd = str_replace ( '__NAME__', static::$PROCESS_NAME, 'ps -ef | grep __NAME__ | grep -v grep | grep -v log | grep -v /usr/bin/php | awk -F \' \' \'{print $2}\'' );
		$output = [ ];
		$ret = 0;
		exec ( $cmd, $output, $ret );
		if ($ret === 0) {
			return $output;
		}
		return false;
	}
	/**
	 * Return true if process is running
	 *
	 * @return boolean
	 */
	public static function isRunning() {
		return sizeof ( self::getPIDs (static::$PROCESS_NAME) ) > 0;
	}
	/**
	 * Stop the running process
	 * @return boolean
	 */
	public static function stop() {
		$pids = self::getPIDs ();
		if (empty ( $pids )) {
			return FALSE;
		}
		foreach ( $pids as $pid ) {
			$output = [ ];
			$ret = 0;
			exec ( 'kill ' . $pid, $output, $ret );
		}
		return true;
	}
	/**
	 * Class used to handle POSIX signals and fork from the current process
	 *
	 * @param string $name
	 *        	<p>The name of the class</p>
	 * @param array $signals
	 *        	<p>An array containing additional POSIX signals to handle [optionel] </p>
	 */
	protected function __construct($name, array $signals = array()) {
		$this->name = $name;
		if (! empty ( $signals )) {
			$this->signals = array_merge ( $this->signals, $signals );
		}
		// Permet au script PHP de s'éxécuter indéfiniment
		set_time_limit ( 0 );
		$this->registerSignals ();
	}
	/**
	 * Used to register POSIX signals
	 */
	private function registerSignals() {
		declare ( ticks = 1 )
			;
		foreach ( $this->signals as $signal ) {
			@pcntl_signal ( $signal, array (
					'Daemon',
					'handleSignal' 
			) );
		}
	}
	/**
	 * Used to handle properly SIGINT, SIGTERM, SIGCHLD and SIGHUP
	 *
	 * @param string $signal        	
	 */
	protected function handleSignal($signal) {
		if ($signal == SIGTERM || $signal == SIGINT) {
			// Gestion de l'extinction
			$this->isRunning = false;
		} else if ($signal == SIGHUP) {
			// Gestion du redémarrage
			$this->onStop ();
			$this->onStart ();
		} else if ($signal == SIGCHLD) {
			// Gestion des processus fils
			pcntl_waitpid ( - 1, $status, WNOHANG );
		} else {
			// Gestion des autres signaux
			$this->handleOtherSignals ( $signal );
		}
	}
	/**
	 * Launch the infinite loop executing the ''run'' abstract method
	 */
	protected function start() {
		$pid = pcntl_fork ();
		if ($pid == - 1) {
			// Erreur
			return false;
		} else if ($pid) {
			// Processus courant (père)
			return true;
		} else {
			// Processus fils (démon)
			// On fait du processus fils un chef de session
			if (($pid = posix_setsid ()) == - 1) {
				// Détachement échoué
				exit ( 1 );
			}
			// On change le nom du processus
			cli_set_process_title ( $this->name );
			// On écrit le PID du processus dans un fichier
			$this->onStart ();
			while ( $this->isRunning ) {
				$this->run ();
			}
			$this->onStop ();
			exit ( 0 );
		}
		exit ( 0 );
	}
	/**
	 * Override to implement the code run infinetly
	 */
	protected abstract function run();
	/**
	 * Override to execute code before the ''run'' method on daemon start
	 */
	protected abstract function onStart();
	/**
	 * Override to execute code after the ''run'' method on daemon shutdown
	 */
	protected abstract function onStop();
	/**
	 * Override to handle additional POSIX signals
	 *
	 * @param int $signal
	 *        	<p>Signal sent by interrupt</p>
	 */
	protected abstract function handleOtherSignals($signal);
}

?>