<?php
class DaoSchedule extends Dao {
	protected static $DB_FILE = 'schedules.db';
	private static $INDEX_ID = 0;
	private static $INDEX_TASK_ID = 1;
	private static $INDEX_TASK_TYPE = 2;
	private static $INDEX_TIME = 3;
	private static $INDEX_ACTIVE = 4;
	/**
	 * Return all scnearii in db
	 *
	 * @return Scneario[]
	 */
	public static function getSchedules() {
		$dbSchedules = self::getValues ();
		if (empty ( $dbSchedules )) {
			return [ ];
		}
		$schedules = [ ];
		foreach ( $dbSchedules as $dbSchedule ) {
			$schedules [] = new Schedule ( $dbSchedule [self::$INDEX_TASK_ID], $dbSchedule [self::$INDEX_TASK_TYPE], $dbSchedule [self::$INDEX_TIME], $dbSchedule [self::$INDEX_ACTIVE], $dbSchedule [self::$INDEX_ID] );
		}
		return $schedules;
	}
	/**
	 * Return the schedule corresponding to the given ID
	 *
	 * @param string $id
	 *        	<br> The schedule id
	 * @return Schedule|boolean <br>The schedule object or false
	 */
	public static function getScheduleByid($id) {
		$dbSchedules = self::getValues ();
		if (! empty ( $dbSchedules )) {
			foreach ( $dbSchedules as $dbSchedule ) {
				if ($dbSchedule [self::$INDEX_ID] == $id) {
					return new Schedule ( $dbSchedule [self::$INDEX_TASK_ID], $dbSchedule [self::$INDEX_TASK_TYPE], $dbSchedule [self::$INDEX_TIME], $dbSchedule [self::$INDEX_ACTIVE], $dbSchedule [self::$INDEX_ID] );
				}
			}
		}
		Logger::warning ( 'Schedule with id ' . $id . ' does not exists' );
		return FALSE;
	}
	/**
	 * Add the schedule to the db
	 *
	 * @param Schedule $schedule        	
	 * @return boolean
	 */
	public static function addSchedule(Schedule $schedule) {
		$dbSchedules = self::getValues ();
		if ($schedule->taskType == Schedule::TASK_TYPE_ACTION) {
			if (DaoAction::getActionById ( $schedule->taskId ) === FALSE) {
				return FALSE;
			}
		} else if ($schedule->taskType == Schedule::TASK_TYPE_SCENARIO) {
			if (DaoScenario::getScenarioByid ( $schedule->taskId ) === FALSE) {
				return FALSE;
			}
		} else if ($schedule->taskType == Schedule::TASK_TYPE_LOGIC) {
			if (DaoLogic::getScenarioByid ( $schedule->taskId ) === FALSE) {
				return FALSE;
			}
		}
		$id = self::getNextId ();
		$schedule->id = $id;
		if ($schedule->active) {
			CronManager::addCron ( $schedule );
		}
		$dbSchedules [] = [ 
				self::$INDEX_ID => $id,
				self::$INDEX_TASK_ID => $schedule->taskId,
				self::$INDEX_TASK_TYPE => $schedule->taskType,
				self::$INDEX_ACTIVE => $schedule->active,
				self::$INDEX_TIME => $schedule->time 
		];
		self::write ( $dbSchedules );
		return $id;
	}
	/**
	 * Modify the schedule
	 *
	 * @param Schedule $schedule        	
	 * @return boolean
	 */
	public static function modifySchedule(Schedule $schedule) {
		$dbSchedules = self::getValues ();
		if ($schedule->taskType == Schedule::TASK_TYPE_ACTION) {
			if (DaoAction::getActionById ( $schedule->taskId ) === FALSE) {
				return FALSE;
			}
		} else if ($schedule->taskType == Schedule::TASK_TYPE_SCENARIO) {
			if (DaoScenario::getScenarioByid ( $schedule->taskId ) === FALSE) {
				return FALSE;
			}
		} else if ($schedule->taskType == Schedule::TASK_TYPE_LOGIC) {
			if (DaoLogic::getScenarioByid ( $schedule->taskId ) === FALSE) {
				return FALSE;
			}
		}
		if ($schedule->active) {
			CronManager::addCron ( $schedule );
		}
		for($i = 0; $i < sizeof ( $dbSchedules ); $i ++) {
			if ($dbSchedules [$i] [self::$INDEX_ID] == $schedule->id) {
				$dbSchedules [$i] [self::$INDEX_TASK_ID] = $schedule->taskId;
				$dbSchedules [$i] [self::$INDEX_TASK_TYPE] = $schedule->taskType;
				$dbSchedules [$i] [self::$INDEX_ACTIVE] = $schedule->active;
				$dbSchedules [$i] [self::$INDEX_TIME] = $schedule->time;
				self::write($dbSchedules);
				return $id;
			}
		}
		return FALSE;
	}
	/**
	 * Remove all schedules with given action ID
	 *
	 * @param int $actionId        	
	 * @return boolean
	 */
	public static function removeSchedulesWithActions($actionId) {
		$dbSchedules = self::getValues ();
		if (empty ( $dbSchedules )) {
			return TRUE;
		}
		for($i = 0; $i < sizeof ( $dbSchedules ); $i ++) {
			if ($dbSchedules [$i] [self::$INDEX_TASK_TYPE] == Schedule::TASK_TYPE_ACTION) {
				if ($dbSchedules [$i] [self::$INDEX_TASK_ID] == $actionId) {
					$scheduleId = $dbSchedules [$i] [self::$INDEX_ID];
					CronManager::deleteCron ( self::getScheduleByid ( $scheduleId ) );
					Logger::debug ( 'Deleting schedule with id ' . $dbSchedules [$i] [self::$INDEX_ID] . 'containing action [ ' . $actionId . ' ]' );
					array_splice ( $dbSchedules, $i, 1 );
				}
			}
		}
		self::write ( $dbSchedules );
		return TRUE;
	}
	/**
	 * Remove all schedules with given scenario ID
	 *
	 * @param int $actionId        	
	 * @return boolean
	 */
	public static function removeSchedulesWithScenario($scenarioId) {
		$dbSchedules = self::getValues ();
		if (empty ( $dbSchedules )) {
			return TRUE;
		}
		for($i = 0; $i < sizeof ( $dbSchedules ); $i ++) {
			if ($dbSchedules [$i] [self::$INDEX_TASK_TYPE] == Schedule::TASK_TYPE_SCENARIO) {
				if ($dbSchedules [$i] [self::$INDEX_TASK_ID] == $scenarioId) {
					$scheduleId = $dbSchedules [$i] [self::$INDEX_ID];
					CronManager::deleteCron ( self::getScheduleByid ( $scheduleId ) );
					Logger::debug ( 'Deleting schedule with id ' . $dbSchedules [$i] [self::$INDEX_ID] . ' containing scenario [ ' . $scenarioId . ' ]' );
					array_splice ( $dbSchedules, $i, 1 );
				}
			}
		}
		self::write ( $dbSchedules );
		return TRUE;
	}
	/**
	 * Remove all schedules with given scenario ID
	 *
	 * @param int $actionId        	
	 * @return boolean
	 */
	public static function removeSchedulesWithLogic($logicId) {
		$dbSchedules = self::getValues ();
		if (empty ( $dbSchedules )) {
			return TRUE;
		}
		for($i = 0; $i < sizeof ( $dbSchedules ); $i ++) {
			if ($dbSchedules [$i] [self::$INDEX_TASK_TYPE] == Schedule::TASK_TYPE_LOGIC) {
				if ($dbSchedules [$i] [self::$INDEX_TASK_ID] == $logicId) {
					$scheduleId = $dbSchedules [$i] [self::$INDEX_ID];
					CronManager::deleteCron ( self::getScheduleByid ( $scheduleId ) );
					Logger::debug ( 'Deleting schedule with id ' . $dbSchedules [$i] [self::$INDEX_ID] . ' containing logic [ ' . $logicId . ' ]' );
					array_splice ( $dbSchedules, $i, 1 );
				}
			}
		}
		self::write ( $dbSchedules );
		return TRUE;
	}
	/**
	 *
	 * @param int $id        	
	 * @param boolean $isActivated        	
	 * @return boolean
	 */
	public static function activateSchedule($id, $isActivated) {
		$dbSchedules = self::getValues ();
		if (empty ( $dbSchedules )) {
			return FALSE;
		}
		for($i = 0; $i < sizeof ( $dbSchedules ); $i ++) {
			if ($dbSchedules [$i] [self::$INDEX_ID] == $id) {
				$dbSchedules [$i] [self::$INDEX_ACTIVE] = $isActivated;
				self::write ( $dbSchedules );
				if ($isActivated) {
					if (! CronManager::addCron ( self::getScheduleByid ( $id ) )) {
						return FALSE;
					}
					Logger::debug ( 'Enabling schedule with id ' . $id );
				} else {
					if (! CronManager::deleteCron ( self::getScheduleByid ( $id ) )) {
					}
					Logger::debug ( 'Disabling schedule with id ' . $id );
				}
				return true;
			}
		}
		Logger::warning ( 'Schedule ' . $id . ' does not exists in db' );
		return FALSE;
	}
	
	/**
	 * Delete an existing schedule
	 *
	 * @param int $id        	
	 * @return boolean
	 */
	public static function delSchedule($id) {
		$dbSchedules = self::getValues ();
		if (empty ( $dbSchedules )) {
			return FALSE;
		}
		for($i = 0; $i < sizeof ( $dbSchedules ); $i ++) {
			if ($dbSchedules [$i] [self::$INDEX_ID] == $id) {
				CronManager::deleteCron ( self::getScheduleByid ( $id ) );
				array_splice ( $dbSchedules, $i, 1 );
				self::write ( $dbSchedules );
				Logger::debug ( 'Deleting schedule with id ' . $id );
				return true;
			}
		}
		Logger::warning ( 'Schedule ' . $id . ' does not exists in db' );
		return FALSE;
	}
}