<?php
class DaoModule extends Dao {
	protected static $DB_FILE = 'module.db';
	private static $INDEX_ID = 0;
	private static $INDEX_IP = 1;
	private static $INDEX_HOSTNAME = 2;
	private static $INDEX_MAC = 3;
	/**
	 * Return all modules in db
	 *
	 * @return EspModule[]
	 */
	public static function getModules() {
		$dbModules = self::getValues ();
		if (empty ( $dbModules )) {
			return [ ];
		}
		$modules = [ ];
		foreach ( $dbModules as $dbModule ) {
			$modules [] = new EspModule ( $dbModule [self::$INDEX_HOSTNAME], $dbModule [self::$INDEX_IP], $dbModule [self::$INDEX_MAC], $dbModule [self::$INDEX_ID] );
		}
		return $modules;
	}
	/**
	 * Return the module corresponding to the given mac address
	 *
	 * @param string $mac
	 *        	<br> The module mac address
	 * @return EspModule|boolean <br>The module object or false
	 */
	public static function getModuleByMac($mac) {
		$dbModules = self::getValues ();
		if (! empty ( $dbModules )) {
			foreach ( $dbModules as $dbModule ) {
				if ($dbModule [self::$INDEX_MAC] == $mac) {
					return new EspModule ( $dbModule [self::$INDEX_HOSTNAME], $dbModule [self::$INDEX_IP], $dbModule [self::$INDEX_MAC], $dbModule [self::$INDEX_ID] );
				}
			}
		}
		Logger::info ( 'Module with mac ' . $mac . ' does not exists' );
		return FALSE;
	}
	/**
	 * Return the module corresponding to the given ip address
	 *
	 * @param string $ip
	 *        	<br> The module IP address
	 * @return EspModule|boolean <br>The module object or false
	 */
	public static function getModuleByIp($ip) {
		$dbModules = self::getValues ();
		if (! empty ( $dbModules )) {
			foreach ( $dbModules as $dbModule ) {
				if ($dbModule [self::$INDEX_IP] == $ip) {
					return new EspModule ( $dbModule [self::$INDEX_HOSTNAME], $dbModule [self::$INDEX_IP], $dbModule [self::$INDEX_MAC], $dbModule [self::$INDEX_ID] );
				}
			}
		}
		Logger::info ( 'Module with IP ' . $ip . ' does not exists' );
		return FALSE;
	}
	/**
	 * Return the module corresponding to the given id
	 *
	 * @param int $id
	 *        	<br> The module id
	 * @return EspModule|boolean <br>The module object or false
	 */
	public static function getModuleById($id) {
		$dbModules = self::getValues ();
		if (! empty ( $dbModules )) {
			foreach ( $dbModules as $dbModule ) {
				if ($dbModule [self::$INDEX_ID] == $id) {
					return new EspModule ( $dbModule [self::$INDEX_HOSTNAME], $dbModule [self::$INDEX_IP], $dbModule [self::$INDEX_MAC], $dbModule [self::$INDEX_ID] );
				}
			}
		}
		Logger::info( 'Module ' . $id . ' does not exists' );
		return FALSE;
	}
	/**
	 * Add the module to the db if not already in
	 *
	 * @param EspModule $module        	
	 * @return boolean
	 */
	public static function addModule(EspModule $module) {
		$m = self::getModuleByMac ( $module->mac );
		if ($m === FALSE) {
			$dbModules = self::getValues ();
			$id = self::getNextId ();
			$dbModules [] = [ 
					self::$INDEX_ID => $id,
					self::$INDEX_HOSTNAME => $module->hostname,
					self::$INDEX_MAC => $module->mac,
					self::$INDEX_IP => $module->ip 
			];
			self::write ( $dbModules );
			Logger::info ( 'Adding module with id ' . $id . ' in db' );
			return $id;
		}
		Logger::info ( 'Module ' . $module->mac . ' already in db' );
		return $m->id;
	}
	/**
	 * Update module IP
	 * @param int $id
	 * @param string $ip
	 * @return boolean
	 */
	public static function updateModuleIp($id, $ip){
		$dbModules = self::getValues ();
		if (empty ( $dbModules )) {
			return FALSE;
		}
		for($i = 0; $i < sizeof ( $dbModules ); $i ++) {
			if ($dbModules [$i] [self::$INDEX_ID] == $id) {
				$dbModules [$i] [self::$INDEX_IP] = $ip;
				self::write ( $dbModules );
				Logger::info ( "Module with id " . $id. ' updated' );
				return TRUE;
			}
		}
		Logger::info( 'Module ' . $id . ' does not exists in db' );
		return FALSE;
	}
	/**
	 * Update module name
	 * @param int $id
	 * @param string $name
	 * @return boolean
	 */
	public static function updateModuleHostname($id, $hostname){
		$dbModules = self::getValues ();
		if (empty ( $dbModules )) {
			return FALSE;
		}
		for($i = 0; $i < sizeof ( $dbModules ); $i ++) {
			if ($dbModules [$i] [self::$INDEX_ID] == $id) {
				$dbModules [$i] [self::$INDEX_HOSTNAME] = $hostname;
				self::write ( $dbModules );
				Logger::info ( "Module with id " . $id. ' updated' );
				return TRUE;
			}
		}
		Logger::info( 'Module ' . $id . ' does not exists in db' );
		return FALSE;
	}
	/**
	 * Delete an existing module
	 *
	 * @param int $id        	
	 * @return boolean
	 */
	public static function delModule($id) {
		$dbModules = self::getValues ();
		if (empty ( $dbModules )) {
			return FALSE;
		}
		for($i = 0; $i < sizeof ( $dbModules ); $i ++) {
			if ($dbModules [$i] [self::$INDEX_ID] == $id) {
				array_splice ( $dbModules, $i, 1 );
				foreach ( DaoAction::getActionsByModuleId ( $id ) as $action ) {
					DaoAction::delAction ( $action->id );
				}
				self::write ( $dbModules );
				Logger::info( "Deleting module with id " . $id );
				return TRUE;
			}
		}
		Logger::info( 'Module ' . $id . ' does not exists in db' );
		return FALSE;
	}
}