#include <ESP8266WiFi.h>
#include <WiFiClient.h>
#include <VS1053.h>
#include <WiFiUDP.h>

// WiFi Parameters //
const char ssid[] = "cfcasa";
const char password[] = "**********";
IPAddress ip(192, 168, 1, 200);
IPAddress gw(192, 168, 1, 254);
IPAddress mask(255, 255, 252, 0);

// MP3 Parameters //
#define VS1053_CS     D1
#define VS1053_DCS    D0
#define VS1053_DREQ   D3
#define VOLUME  70 // volume level 0-100
VS1053 player(VS1053_CS, VS1053_DCS, VS1053_DREQ);
uint8_t volume_level = VOLUME;

// MP3 Server parameters //
const uint16_t MP3_PORT = 5045;
WiFiServer server(MP3_PORT);
const uint16_t BUFFER_SIZE = 16384;
//const uint16_t BUFFER_SIZE = 32768;
byte buffer[BUFFER_SIZE + 1];
uint16_t byteRead = 0;
bool isBufferReady = false;
WiFiClient client;

// Command Server parameters //
const uint16_t CMD_PORT = 5046;
// L'instance du serveur UDP
WiFiUDP udp;
// Liste de commandes
const char CMD_VOLUME = 'V';
const char CMD_LOCATE = 'L';
const char CMD_RESET = 'R';
const char RESPONSE_NOK[] = "N";

// Localisation parameters
uint8_t ledPin = LED_BUILTIN;
const uint16_t BLINK_FREQ = 500;
const uint8_t BLINK_REP = 60;
uint8_t repetition = 0;
uint32_t timer;
bool isLocate = false;

void setup() {
  // on démarre le port série
  Serial.begin(115200);
  // On attend "un peu" que le buffer soit prêt
  delay(10);
  // On démarre le WiFi
  initWiFi();
  // On démarre le shield MP3
  initMP3();
  // On démarre le serveur TCP
  initMp3Server();
  // On démarre le serveur UDP
  initCmdServer();
  // On prépare la led de localisation
  initLocateLed();
}

void loop() {
  readMp3();
  readCmd();
}

void readMp3() {
  // Un client est connecté
  if (client) {
    // Si le client est toujours connecté...
    if (client.connected()) {
      // Tant que l'on reçoit de la musique
      while (client.available() > 0) {
        // On rempli le buffer avec le contenu du paquet TCP
        buffer[byteRead++] = client.read();
        // Si le buffer est plein
        if (byteRead == BUFFER_SIZE) {
          // On previent que le buffer est plein
          isBufferReady = true;
          // On sort de la boucle
          break;
        }
      }
      // Si le buffer est plein
      if (isBufferReady) {
        // On envoi tout au lecteur MP3
        player.playChunk(buffer, byteRead);
        // On RAZ le compteur de bytes
        byteRead = 0;
      }
    } else {
      // Si le client est déconneté
      client = server.available();
      // On prévient que le buffer est vide
      isBufferReady = false;
      // On RAZ le compteur de bytes
      byteRead = 0;
      Serial.println(F("Client disconnected"));
    }
  } else {
    // On test la présence d'un client
    client = server.available();
    if (client) {
      // Un client est connecté !
      Serial.println(F("New client !"));
    }
  }
}

bool readCmd() {
  if (udp.parsePacket() > 0) {
    uint8_t len = udp.available();
    char c = udp.read();
    if (len == 1) {
      if (c == CMD_RESET) {
        player.softReset();
        sendPacket(CMD_RESET);
        return true;
      } else if (c == CMD_VOLUME) {
        sendPacket(volume_level);
        return true;
      } else if (c == CMD_LOCATE) {
        if (!isLocate) {
          locate(true);
          sendPacket(CMD_LOCATE);
          return true;
        }
      }
    } else if (len == 2) {
      char value = udp.read();
      if (c == CMD_VOLUME) {
        if (value == '+' && volume_level != 100) {
          setVolume(volume_level + 1);
        } else if (value == '-' && volume_level != 0) {
          setVolume(volume_level - 1);
        } else {
          sendNack();
          return false;
        }
        return true;
      }
    } else {
      if (c == CMD_VOLUME) {
        char value[4] = "0";
        udp.read(value, 3);
        uint8_t vol = atoi(value);
        if (vol >= 0 && vol <= 100) {
          setVolume(vol);
          return true;
        }
      }
    }
    sendNack();
    return false;
  }
  // Check if locate needs to be done
  locate(false);
}

void locate(bool blink) {
  if (blink) {
    // Initialisation des variables
    digitalWrite(ledPin, LOW);
    isLocate = true;
    repetition = 0;
    timer = millis();
  } else if (isLocate) {
    // Calcule du temps
    uint32_t elapsed = millis() - timer;
    if (elapsed > BLINK_FREQ) {
      repetition++;
      if (repetition == BLINK_REP) {
        // On a atteint le nombre de clignotements
        isLocate = false;
        digitalWrite(ledPin, HIGH);
      } else {
        // On fait clignoter la led
        digitalWrite(ledPin, !digitalRead(ledPin));
        timer = millis();
      }
    }
  }
}

void setVolume(uint8_t vol) {
  volume_level = vol;
  Serial.printf("Setting volume to %d\n", volume_level);
  player.setVolume(volume_level);
  sendPacket(volume_level);
}

void sendNack() {
  sendPacket(RESPONSE_NOK);
}

void sendPacket(const char content) {
  udp.beginPacket(udp.remoteIP(), udp.remotePort());
  udp.write(content);
  udp.endPacket();
}

void sendPacket(const char content[]) {
  udp.beginPacket(udp.remoteIP(), udp.remotePort());
  udp.write(content);
  udp.endPacket();
}

void sendPacket(int content) {
  char value[4];
  udp.beginPacket(udp.remoteIP(), udp.remotePort());
  udp.write(itoa(content, value, 10));
  udp.endPacket();
}

void initWiFi() {
  // On efface la configuration précédente
  WiFi.disconnect(true);
  Serial.printf("\nConnexion a %s", ssid);
  // Initialisation de la connection
  WiFi.config(ip, gw, mask, gw);
  WiFi.begin(ssid, password);
  // Test pour déterminer quand la connection est prete
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  // Affichage des informations
  Serial.printf(" connecté\nAdresse IP: %s\n", WiFi.localIP().toString().c_str());
}

void initMP3() {
  SPI.begin();
  // Démarrage du lecteur MP3
  player.begin();
  player.switchToMp3Mode();
  player.setVolume(volume_level);
}

void initMp3Server() {
  // Démarrage du serveur TCP
  server.begin();
  Serial.printf("Ecoute TCP sur le port %d\n", MP3_PORT);
}

void initCmdServer() {
  // Démarrage du serveur TCP
  udp.begin(CMD_PORT);
  Serial.printf("Ecoute UDP sur le port %d\n", CMD_PORT);
}

void initLocateLed() {
  pinMode(ledPin, OUTPUT);
  digitalWrite(ledPin, HIGH);
}

