<?php
/**
 *
 * @author Tala
 *
 */
class Logger {
	const LOG_LEVEL_OFF = 5;
	const LOG_LEVEL_FATAL = 4;
	const LOG_LEVEL_ERROR = 3;
	const LOG_LEVEL_WARNING = 2;
	const LOG_LEVEL_INFO = 1;
	const LOG_LEVEL_DEBUG = 0;
	private static $LOG_STRINGS = ['DEBUG', 'INFO', 'WARNING', 'ERROR', 'FATAL'];
	// Dossier de journalisation par défaut
	private $logDir = __DIR__; 
	// Fichier de journalisation par défaut
	private $logFile = 'log/domotech.log';
	// Niveau de journalisation par défaut
	private $logLevel = self::LOG_LEVEL_DEBUG;
	// Instance de Logger
	private static $instance;
	// Dernier message du Logger
	private $lastLog;
	/**
	 * Used to retrieve the Logger instance
	 *
	 * @return Logger
	 */
	private static function getInstance() {
		if (self::$instance == NULL) {
			self::$instance = new Logger ();
		}
		return self::$instance;
	}
	/**
	 * Used to change the log directory
	 *
	 * @param string $filename
	 */
	public static function setLogDirectory($directory) {
		self::getInstance ()->logDir = $directory;
	}
	/**
	 * Used to change the log file
	 *
	 * @param string $filename        	
	 */
	public static function setLogFile($filename) {
		self::getInstance ()->logFile = $filename;
	}
	/**
	 * Used to change the log level
	 *
	 * @param int $logLevel        	
	 */
	public static function setLogLevel($logLevel) {
		self::getInstance ()->logLevel = $logLevel;
	}
	/**
	 * Log an event at a debug level
	 *
	 * @param string $message
	 *        	The event to log
	 * @return boolean TRUE if event is logged, FALSE otherwise
	 */
	public static function debug($message, $caller = ""){
		self::log($message, self::LOG_LEVEL_DEBUG, $caller);
	}
	/**
	 * Log an event at an info level
	 *
	 * @param string $message
	 *        	The event to log
	 * @return boolean TRUE if event is logged, FALSE otherwise
	 */
	public static function info($message, $caller = ""){
		self::log($message, self::LOG_LEVEL_INFO, $caller);
	}
	/**
	 * Log an event at a warning level
	 *
	 * @param string $message
	 *        	The event to log
	 * @return boolean TRUE if event is logged, FALSE otherwise
	 */
	public static function warning($message, $caller = ""){
		self::log($message, self::LOG_LEVEL_WARNING, $caller);
	}
	/**
	 * Log an event at an error level
	 *
	 * @param string $message
	 *        	The event to log
	 * @return boolean TRUE if event is logged, FALSE otherwise
	 */
	public static function error($message, $caller = ""){
		self::log($message, self::LOG_LEVEL_ERROR, $caller);
	}
	/**
	 * Log an event at a fatal level
	 *
	 * @param string $message
	 *        	The event to log
	 * @return boolean TRUE if event is logged, FALSE otherwise
	 */
	public static function fatal($message, $caller = ""){
		self::log($message, self::LOG_LEVEL_FATAL, $caller);
	}
	/**
	 * Log an event according to its level
	 *
	 * @param string $message
	 *        	The event to log
	 * @param int $level
	 *        	The event level, DEBUG by default
	 * @return boolean TRUE if event is logged, FALSE otherwise
	 */
	public static function log($message, $level = self::LOG_LEVEL_DEBUG, $caller = "") {
		self::getInstance()->lastLog = $message;
		// On vérifie si l'événement doit être journalisé
		if ($level < self::getInstance ()->logLevel || $level === self::LOG_LEVEL_OFF) {
			return false;
		}
		// On respect le format de journalisation ''syslog''
		$log = date ( "M j H:i:s" ) . ' : ';
		$log .= self::getLogString($level).' : ';
		if ($level === self::LOG_LEVEL_DEBUG && empty ( $caller )) {
			$callerId = sizeof ( debug_backtrace () ) - 1 < 0 ? 0 : sizeof ( debug_backtrace () ) - 1;
			if (isset ( debug_backtrace () [$callerId] ['class'] )) {
				$caller = debug_backtrace () [$callerId] ['class'] . "->" . debug_backtrace () [$callerId] ['function'];
			}
		}
		if (! empty ( $caller )) {
			$log .= "$caller : ";
		}
		if(!empty($_SERVER["REMOTE_ADDR"])){
			$log .= $_SERVER["REMOTE_ADDR"] . " : ";
		}
		$log .= trim ( $message ) . "\n";
		if (! file_put_contents ( self::getInstance ()->logFile, $log, FILE_APPEND )) {
			error_log ( "Cannot write file " . self::getInstance ()->logFile );
		}
		return true;
	}
	/**
	 * Return the last logged message
	 * 
	 * @return string
	 */
	public static function getLastMessage() {
		return self::getInstance ()->lastLog;
	}
	/**
	 * Return the string representation of the given log level
	 * @param int $level
	 * @return boolean|string
	 */
	private static function getLogString($level){
		if($level > sizeof(self::$LOG_STRINGS)-1){
			return FALSE;
		}
		return self::$LOG_STRINGS[$level];
	}
}