<?php
/**
 * Used to access modules
 * @author Tala
 *
 */
class ModuleManager {
	
	/**
	 * Retrieve all neighbours names
	 *
	 * @return boolean|array[]
	 */
	public static function retrieveNeighbours() {
		$socket = SocketHelper::getMulticastSocket ( Configuration::MULTICAST_IP );
		if (SocketHelper::sendUdpPacket ( Configuration::MULTICAST_MSG_INFO, Configuration::MULTICAST_IP, Configuration::MULTICAST_PORT, $socket ) === FALSE) {
			return false;
		}
		$modules = [ ];
		$start = microtime ( true );
		$remaining = 0;
		while ( $remaining < 4000 ) {
			$buffer = SocketHelper::readUdpPacket ( $socket, $ip, $port, 500 );
			if (! empty ( $buffer )) {
				Logger::debug ( 'Packet received from ' . $ip . ':' . $port . ' [ ' . $buffer . ' ]' );
				$hostname = strpos ( $buffer, Configuration::MULTICAST_MSG_INFO . ':' ) !== FALSE ? substr ( $buffer, strlen ( Configuration::MULTICAST_MSG_INFO ) + 1 ) : $buffer;
				$mac = self::getModuleMacFromARP ( $ip );
				$modules [] = new EspModule ( $hostname, $ip, $mac );
			}
			$remaining = (microtime ( true ) - $start) * 1000;
		}
		socket_close ( $socket );
		return $modules;
	}
	/**
	 * Get the configuration of the ESP module
	 *
	 * @param string $ip        	
	 * @return boolean|string
	 */
	public static function retrieveConfiguration($ip) {
		$socket = SocketHelper::getMulticastSocket ( Configuration::MULTICAST_IP );
		if (SocketHelper::sendUdpPacket ( Configuration::MULTICAST_MSG_CONF, $ip, Configuration::MULTICAST_PORT, $socket ) === FALSE) {
			return false;
		}
		$buffer = SocketHelper::readUdpPacket ( $socket, $sip, $sport, 2000 );
		if (! empty ( $buffer )) {
			Logger::debug ( 'Packet received from ' . $sip . ':' . $sport . ' [ ' . $buffer . ' ]' );
			return strpos ( $buffer, Configuration::MULTICAST_MSG_CONF . ':' ) !== FALSE ? substr ( $buffer, strlen ( Configuration::MULTICAST_MSG_CONF ) + 1 ) : $buffer;
		}
		socket_close ( $socket );
		return FALSE;
	}
	/**
	 * Used to adapt module Ip address accordingly
	 */
	public static function checkModules(&$updated, &$missing, &$unchanged) {
		// Used to refresh ARP table
		self::retrieveNeighbours ();
		$modules = DaoModule::getModules ();
		foreach ( $modules as $module ) {
			$ip = self::getModuleIpFromARP ( $module->mac );
			$configuration = self::retrieveConfiguration ( $ip );
			$hostname = '';
			if ($configuration !== FALSE) {
				$hostname = self::getModuleHostnameFromConfiguration ( $configuration );
			}
			if ($ip == FALSE) {
				$missing [] = $module->id;
			} else if ($ip !== $module->ip || $hostname != $module->hostname) {
				$updated [] = $module->id;
				DaoModule::updateModuleIp ( $module->id, $ip );
				if (! empty ( $hostname )) {
					DaoModule::updateModuleHostname ( $module->id, $hostname );
				}
			} else {
				$unchanged [] = $module->id;
			}
		}
		return TRUE;
	}
	/**
	 * Extract the mac address from the module configuration
	 *
	 * @param string $configuration
	 *        	<br> the json configuration
	 * @return string|boolean
	 */
	public static function getModuleMacFromConfiguration($configuration) {
		$conf = json_decode ( $configuration );
		return $conf->configuration->mac;
	}
	/**
	 * Extract the mac address from the module configuration
	 *
	 * @param string $configuration
	 *        	<br> the json configuration
	 * @return string|boolean
	 */
	public static function getModuleHostnameFromConfiguration($configuration) {
		$conf = json_decode ( $configuration );
		return $conf->configuration->hostname;
	}
	/**
	 * Extract the mac address from the ARP table
	 *
	 * @param string $configuration
	 *        	<br> the json configuration
	 * @return string|boolean
	 */
	public static function getModuleMacFromARP($ip) {
		$output = [ ];
		$ret = 0;
		exec('ping -c 1 '.$ip, $output, $ret);
		if ($ret !== 0) {
			Logger::error ( 'Fail pinging '.$ip );
			return FALSE;
		}
		$cmd = 'arp -a ' . $ip;
		exec ( $cmd, $output, $ret );
		if ($ret !== 0) {
			Logger::error ( 'Fail retrieving MAC from ARP' );
			return FALSE;
		}
		$output = $output [sizeof ( $output ) - 1];
		if (preg_match ( '@\s([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})\s@', $output, $matches ) !== 1) {
			Logger::error ( 'Fail retrieving MAC from [ ' . $output . ' ], command was [ '.$cmd.' ]' );
			return FALSE;
		}
		$mac = strpos ( $matches [0], '-' ) !== FALSE ? str_replace ( '-', ':', $matches [0] ) : $matches [0];
		return trim ( $mac );
	}
	/**
	 * Extract the mac address from the ARP table
	 *
	 * @param string $configuration
	 *        	<br> the json configuration
	 * @return string|boolean
	 */
	public static function getModuleIpFromARP($mac) {
		$output = [ ];
		$ret = 0;
		exec ( 'arp -a', $output, $ret );
		if ($ret !== 0 || empty ( $output )) {
			Logger::error ( 'Fail retrieving IP from ARP' );
			return FALSE;
		}
		if (PHP_OS == 'WINNT') {
			$mac = str_replace ( ':', '-', $mac );
		}
		foreach ( $output as $line ) {
			if (strpos ( $line, $mac ) != FALSE) {
				$regex = '@\((.*)\)\s@';
				if (PHP_OS == 'WINNT') {
					$regex = '@\s((\d{1,3}.){3}.\d{1,3})\s@';
				}
				if (! preg_match ( $regex, $line, $matches )) {
					Logger::error ( 'Fail retrieving IP from [ ' . $line . ' ]' );
					return FALSE;
				}
				$ip = strpos ( $matches [1], '-' ) !== FALSE ? str_replace ( '-', ':', $matches [1] ) : $matches [1];
				return trim ( $ip );
			}
		}
		return FALSE;
	}
	/**
	 * Retrieve the gpio states
	 *
	 * @param string $ip
	 *        	<br> the module ip
	 * @return string
	 */
	public static function getGpioStates($ip) {
		$url = 'http://' . $ip . 'api/gpio';
		return self::sendHttpRequest ( $url );
	}
	/**
	 * Retrieve the gpio states
	 *
	 * @param string $ip
	 *        	<br> the module ip
	 * @return string
	 */
	public static function getGpioState($ip, $gpio) {
		$url = 'http://' . $ip . '/api/gpio/state?gpio=' . $gpio;
		$json = self::sendHttpRequest ( $url );
		if ($json === FALSE) {
			return FALSE;
		}
		return $json;
	}
	/**
	 * Modify a gpio state
	 *
	 * @param string $ip
	 *        	<br> the module ip
	 * @param int $gpio
	 *        	<br> the gpio name (D0, D1, ...)
	 * @param bool $state
	 *        	<br> the desired state
	 */
	public static function modifyGpioState($ip, $gpio, $state) {
		$url = 'http://' . $ip . '/api/gpio/state?gpio=' . $gpio . '&state=';
		$url .= $state ? '1' : '0';
		return self::sendHttpRequest ( $url );
	}
	/**
	 * Toggle a gpio
	 *
	 * @param string $ip        	
	 * @param int $gpio        	
	 * @return string
	 */
	public static function toggleGpioState($ip, $gpio) {
		$url = 'http://' . $ip . '/api/gpio/toggle?gpio=' . $gpio;
		return self::sendHttpRequest ( $url );
	}
	/**
	 * Time a gpio state
	 *
	 * @param string $ip        	
	 * @param int $gpio        	
	 * @return string
	 */
	public static function timeGpioState($ip, $gpio, $time) {
		$url = 'http://' . $ip . '/api/gpio/time?gpio=' . $gpio . '&time=' . $time;
		return self::sendHttpRequest ( $url );
	}
	/**
	 * Apply PWM a gpio
	 *
	 * @param string $ip        	
	 * @param int $gpio        	
	 * @return string
	 */
	public static function pwmGpio($ip, $gpio, $pwm) {
		$url = 'http://' . $ip . '/api/gpio/pwm?gpio=' . $gpio . '&pwm=' . $pwm;
		return self::sendHttpRequest ( $url );
	}
	/**
	 * Read temperature
	 *
	 * @param string $ip        	
	 * @param int $gpio        	
	 * @return string
	 */
	public static function readTemp($ip) {
		$url = 'http://' . $ip . '/api/temp';
		return self::sendHttpRequest ( $url );
	}
	/**
	 * Send an http request
	 *
	 * @param string $ip        	
	 * @param string $url        	
	 * @return string
	 */
	private static function sendHttpRequest($url) {
		$code = 0;
		$ret = HttpHelper::sendHttpRequest ( $url, $code );
		if ($code != 200) {
			Logger::warning ( 'Fail sending http request' );
		}
		return $ret;
	}
}